/**
 * TypeScript definitions for Kore Payment SDK
 */

// Error classes
export class KoreError extends Error {
  code: string;
  details: any;
  constructor(message: string, code: string, details?: any);
}

export class KoreAuthenticationError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KoreValidationError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KorePaymentError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KoreNetworkError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KoreRateLimitError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KoreNotFoundError extends KoreError {
  constructor(message?: string, details?: any);
}

export class KoreServerError extends KoreError {
  constructor(message?: string, details?: any);
}

// Configuration
export interface KoreConfig {
  apiKey: string;
  baseUrl?: string;
  environment?: 'sandbox' | 'live';
  timeout?: number;
  retryAttempts?: number;
  retryDelay?: number;
  onError?: (error: KoreError) => void;
  onRequest?: (config: RequestConfig) => void;
  onResponse?: (response: Response) => void;
  debug?: boolean;
  logger?: (message: string) => void;
}

export interface RequestConfig {
  method: string;
  headers: Record<string, string>;
  body?: string;
  [key: string]: any;
}

// Session data
export interface SessionData {
  order_id: string;
  amount: number;
  currency: string;
  capture_mode?: 'auto' | 'manual';
  payer: {
    email: string;
    name: string;
    phone?: string;
  };
  billing: {
    address_line_1: string;
    address_line_2?: string;
    city: string;
    state?: string;
    postal_code: string;
    country: string;
  };
  shipping?: {
    address_line_1: string;
    address_line_2?: string;
    city: string;
    state?: string;
    postal_code: string;
    country: string;
    name?: string;
    phone?: string;
  };
  return_url: string;
  cancel_url: string;
  metadata?: Record<string, any>;
  processing_channel_id?: string; // Optional field for Checkout.com only
}

export interface SessionOptions {
  idempotencyKey?: string;
  timeout?: number;
  [key: string]: any;
}

export interface SessionResponse {
  session_id: string;
  payment_id: string;
  redirect_url: string;
  expires_at: string;
}

export interface SessionStatus {
  session_id: string;
  status: 'pending' | 'requires_action' | 'redirected' | 'completed' | 'failed';
  payment_id: string | null;
  order_id: string;
  amount: number;
  currency: string;
  created_at: string;
  expires_at: string;
}

// Transaction
export interface Transaction {
  payment_id: string;
  order_id: string;
  amount: number;
  currency: string;
  status: 'pending' | 'authorized' | 'captured' | 'failed' | 'refunded' | 'partial_refunded' | 'canceled';
  gateway: {
    id: number;
    code: string;
    name: string;
  } | null;
  gateway_payment_id: string | null;
  authorized_at: string | null;
  captured_at: string | null;
  failed_at: string | null;
  failure_reason: string | null;
  created_at: string;
  updated_at: string;
}

// Redirect
export interface RedirectParams {
  payment_id?: string;           // Optional, may be null for errors
  order_id: string;
  status: 'captured' | 'authorized' | 'failed' | 'canceled' | 'pending';
  gateway?: string;              // Optional gateway code
  error_code?: string;           // Present if status is 'failed'
  error_message?: string;        // Present if status is 'failed'
  signature: string;
}

export interface RedirectResult {
  success: boolean;
  data?: {
    payment_id: string;
    order_id: string;
    status: 'captured' | 'authorized' | 'failed' | 'canceled' | 'pending';
    gateway?: string;
  };
  error?: {
    code: string;
    message: string;
    payment_id?: string;
    order_id?: string;
    gateway?: string;
  };
}

export interface RedirectOptions {
  target?: '_self' | '_blank';
}

// Polling
export interface PollOptions {
  interval?: number;
  maxAttempts?: number;
  onStatusChange?: (status: SessionStatus) => void;
}

// Error Codes
export const ErrorCodes: {
  // SDK Errors
  AUTHENTICATION_ERROR: string;
  VALIDATION_ERROR: string;
  PAYMENT_ERROR: string;
  NETWORK_ERROR: string;
  RATE_LIMIT_ERROR: string;
  NOT_FOUND_ERROR: string;
  SERVER_ERROR: string;
  // Redirect Errors
  SESSION_NOT_FOUND: string;
  PAYMENT_NOT_FOUND: string;
  GATEWAY_NOT_AVAILABLE: string;
  GATEWAY_CREDENTIALS_ERROR: string;
  GATEWAY_ADAPTER_ERROR: string;
  GATEWAY_SESSION_CREATION_FAILED: string;
  GATEWAY_VERIFICATION_FAILED: string;
  GATEWAY_TIMEOUT: string;
  NO_ACTIVE_API_KEY: string;
  RESOURCE_NOT_FOUND: string;
  BAD_REQUEST: string;
  FORBIDDEN: string;
  INTERNAL_ERROR: string;
  UNKNOWN_ERROR: string;
};

// Main SDK class
export class Kore {
  constructor(config: KoreConfig);
  createSession(data: SessionData, options?: SessionOptions): Promise<SessionResponse>;
  getSessionStatus(sessionId: string): Promise<SessionStatus>;
  getTransaction(paymentId: string | number): Promise<Transaction>;
  redirectToPayment(redirectUrl: string, options?: RedirectOptions): void;
  handleRedirectReturn(urlParams: URLSearchParams | Record<string, string>, apiSecret: string): Promise<RedirectResult>;
  pollPaymentStatus(sessionId: string, options?: PollOptions): Promise<SessionStatus>;
  verifyRedirectSignature(params: RedirectParams, apiSecret: string): boolean;
  verifyRedirectSignatureAsync(params: RedirectParams, apiSecret: string): Promise<boolean>;
  static readonly ErrorCodes: typeof ErrorCodes;
  static generateIdempotencyKey(): string;
  static formatAmount(amount: number, currency: string): string;
  static validateCurrency(currency: string): boolean;
  static validateEmail(email: string): boolean;
}

export default Kore;

// Utility functions
export function generateIdempotencyKey(): string;
export function formatAmount(amount: number, currency: string): string;
export function validateCurrency(currency: string): boolean;
export function validateEmail(email: string): boolean;

